#include "bmi_exported.h"
#include "bmi_mesgerr.h"
#include "bmi_indices.h"
#if ! defined (BMI_FLAT_DIR)
#    include "exported/bmi_equations.h"
#else
#    include "bmi_equations.h"
#endif


static void bmi_printf_rewrite_rules_mpz 
				(bap_tableof_polynom_mpz T, char po, char pf)
{   struct bav_rank rg;
    struct bap_polynom_mpz init, red;
    ba0_int_p i;

    bap_init_readonly_polynom_mpz (&init);
    bap_init_readonly_polynom_mpz (&red);
    ba0_put_char (po);
    for (i = 0; i < T->size; i++)
    {	rg = bap_rank_polynom_mpz (T->tab [i]);
	bap_initial_and_reductum_polynom_mpz (&init, &red, T->tab [i]);
	ba0_printf ("%rank = - (%Az)/(%Az)", &rg, &red, &init);
	if (i < T->size-1)
	    ba0_put_string (", ");
    }
    ba0_put_char (pf);
}

static void bmi_printf_rewrite_rules_mpq 
				(bap_tableof_polynom_mpq T, char po, char pf)
{   struct bav_rank rg;
    struct bap_polynom_mpq init, red;
    ba0_int_p i;

    bap_init_readonly_polynom_mpq (&init);
    bap_init_readonly_polynom_mpq (&red);
    ba0_put_char (po);
    for (i = 0; i < T->size; i++)
    {	rg = bap_rank_polynom_mpq (T->tab [i]);
	bap_initial_and_reductum_polynom_mpq (&init, &red, T->tab [i]);
	ba0_printf ("%rank = - (%Aq)/(%Aq)", &rg, &red, &init);
	if (i < T->size-1)
	    ba0_put_string (", ");
    }
    ba0_put_char (pf);
}

/*
 * EXPORTED
 * Equations (regchain, fullset)
 *
 * Returns the sequence of the list of equations for each regchain.
 * At least one regchain is mandatory.
 *
 * fullset = true | false
 */

static ALGEB bmi__equations (struct bmi_callback* callback, bool rw)
{   struct bad_regchain C;
    bool fullset;
    char *buffer;

    if (bmi_nops (callback) != 2)
	BA0_RAISE_EXCEPTION (BMI_ERRNOPS);
    if (! bmi_is_regchain_op (1, callback))
            BA0_RAISE_EXCEPTION (BMI_ERRREGC);

    bmi_set_ordering_and_regchain (&C, 1, callback, __FILE__, __LINE__);
    fullset = bmi_bool_op (2, callback);
    if (! fullset)
	bad_remove_zero_derivatives_of_tableof_parameter_from_regchain 
						(&C, &C, &bav_parameters);

    bav_set_settings_symbol (0, &bav_printf_numbered_symbol);
    if (rw)
    {	ba0_record_output ();
	ba0_set_output_counter ();
	bmi_printf_rewrite_rules_mpz (&C.decision_system, '[', ']');
	buffer = ba0_persistent_malloc (ba0_output_counter () + 1);
	ba0_set_output_string (buffer);
	bmi_printf_rewrite_rules_mpz (&C.decision_system, '[', ']');
	ba0_restore_output ();
    } else
	buffer = ba0_new_printf ("%t[%Az]", &C.decision_system);

    {   ALGEB res;
	bmi_push_maple_gmp_allocators ();
        res = EvalMapleStatement (callback->kv, buffer);
	bmi_pull_maple_gmp_allocators ();
        return res;
    }
}


ALGEB bmi_equations (struct bmi_callback* callback)
{
    return bmi__equations (callback, false);
}

ALGEB bmi_rewrite_rules (struct bmi_callback* callback)
{
    return bmi__equations (callback, true);
}

/*
 * Subfunctions of EquationWithLeaderOrOrder
 */

static bool bmi_test_leader_eq (bav_variable v, bool numeric, bav_variable u)
{
    return (!numeric) && u == v; 
}

static bool bmi_test_rank_eq (bav_rank rg, bav_rank rk)
{
    return bav_equal_rank (rk, rg);
}

static bool bmi_test_order_eq (bav_Iorder o, bool indep, bav_Iorder p)
{
    return (!indep) && o == p;
}

static bool bmi_test_leader_ne (bav_variable v, bool numeric, bav_variable u)
{
    return (!numeric) && u != v;
}

static bool bmi_test_rank_ne (bav_rank rg, bav_rank rk)
{
    return ! bav_equal_rank (rk, rg);
}

static bool bmi_test_order_ne (bav_Iorder o, bool indep, bav_Iorder p)
{
    return (!indep) && o != p;
}

static bool bmi_test_leader_gt (bav_variable v, bool numeric, bav_variable u)
{  
    return (!numeric) && bav_R_variable_number (u) > bav_R_variable_number (v);
}

static bool bmi_test_rank_gt (bav_rank rg, bav_rank rk)
{
    return bav_gt_rank (rk, rg);
}

static bool bmi_test_order_gt (bav_Iorder o, bool indep, bav_Iorder p)
{
    return (!indep) && p > o;
}

static bool bmi_test_leader_ge (bav_variable v, bool numeric, bav_variable u)
{ 
    return (!numeric) && 
		bav_R_variable_number (u) >= bav_R_variable_number (v);
}

static bool bmi_test_rank_ge (bav_rank rg, bav_rank rk)
{
    return bav_equal_rank (rk, rg) || bav_gt_rank (rk, rg);
}

static bool bmi_test_order_ge (bav_Iorder o, bool indep, bav_Iorder p)
{
    return (!indep) && p >= o;
}

static bool bmi_test_leader_lt (bav_variable v, bool numeric, bav_variable u)
{ 
    return (!numeric) && bav_R_variable_number (u) < bav_R_variable_number (v);
}

static bool bmi_test_rank_lt (bav_rank rg, bav_rank rk)
{
    return bav_lt_rank (rk, rg);
}

static bool bmi_test_order_lt (bav_Iorder o, bool indep, bav_Iorder p)
{
    return (!indep) && p < o;
}

static bool bmi_test_leader_le (bav_variable v, bool numeric, bav_variable u)
{ 
    return (!numeric) && 
		bav_R_variable_number (u) <= bav_R_variable_number (v);
}

static bool bmi_test_rank_le (bav_rank rg, bav_rank rk)
{
    return bav_equal_rank (rk, rg) || bav_lt_rank (rk, rg);
}

static bool bmi_test_order_le (bav_Iorder o, bool indep, bav_Iorder p)
{
    return (!indep) && p <= o;
}

static bool bmi_test_leader_deriv_eq 
			(bav_variable v, bool numeric, bav_variable u)
{ 
    return (!numeric) && u->root->type != bav_independent_symbol
						&& bav_is_derivative (u, v);
}

static bool bmi_test_leader_deriv_ne 
			(bav_variable v, bool numeric, bav_variable u)
{
    return (!numeric) && (u->root->type == bav_independent_symbol
						|| !bav_is_derivative (u, v));
}

static bool bmi_test_leader_proper_eq 
			(bav_variable v, bool numeric, bav_variable u)
{ 
    return (!numeric) && u->root->type != bav_independent_symbol
				&& u != v && bav_is_derivative (u, v);
}

static bool bmi_test_leader_proper_ne 
			(bav_variable v, bool numeric, bav_variable u)
{
    return (!numeric) && (u->root->type == bav_independent_symbol
				|| u == v || !bav_is_derivative (u, v));
}

/*
 * EXPORTED
 * This function is actually called through the Equations function
 * of the DifferentialAlgebra package.
 *
 * EquationsWithCriterion 
 * 	(list(polynomial) | regchain, 
 * 	 keyword, relop, modifier, derv, fullset, differential ring)
 *
 * Returns a subsequence of the list of equations
 */

static ALGEB bmi__equations_with_criterion 
				(struct bmi_callback* callback, bool rw)
{   struct bad_regchain C;
    ba0_table T;
    struct bav_rank rg;
    bav_variable v;
    bav_Iorder o;
    bool fullset;
    char *eqns, *keyword, *relop, *modifier, *derv;
    char *buffer;
    bool (*fleader) (bav_variable, bool, bav_variable);
    bool (*forder) (bav_Iorder, bool, bav_Iorder);
    bool (*frank) (bav_rank, bav_rank);
    bool leader, rank, order, integer;
    ba0_int_p i;

    fleader = 0;
    frank = 0;
    forder = 0;
    o = 0;		/* to avoid a warning */

    if (bmi_nops (callback) != 7)
	BA0_RAISE_EXCEPTION (BMI_ERRNOPS);
    if (! bmi_is_table_op (7, callback))
	BA0_RAISE_EXCEPTION (BMI_ERRDRNG);

    if (bmi_is_regchain_op (1, callback))
	bmi_set_ordering_and_regchain (&C, 1, callback, __FILE__, __LINE__);
    else
	bmi_set_ordering (7, callback, __FILE__, __LINE__);

    keyword = bmi_string_op (2, callback);
    relop = bmi_string_op (3, callback);
    modifier = bmi_string_op (4, callback);
    derv = bmi_string_op (5, callback);

    if (bmi_is_regchain_op (1, callback))
    {	fullset = bmi_bool_op (6, callback);
	if (! fullset)
	    bad_remove_zero_derivatives_of_tableof_parameter_from_regchain 
						(&C, &C, &bav_parameters);
	integer = true;
	T = (ba0_table)&C.decision_system;
    }
    else
    {	eqns = bmi_string_op (1, callback);
	integer = false;
	T = ba0_new_table ();
	ba0_sscanf2 (eqns, "%t[%careful_expanded_Aq]", T);
    }

    leader = rank = order = false;
    if (strcmp (keyword, BMI_IX_leader) == 0)
    {	leader = true;
	ba0_sscanf2 (derv, "%v", &v);
    } else if (strcmp (keyword, BMI_IX_order) == 0)
    {	order = true;
	o = (bav_Iorder)atoi (derv);
    } else if (strcmp (keyword, BMI_IX_rank) == 0)
    {	rank = true;
	ba0_sscanf2 (derv, "%rank", &rg);
    } else
	BA0_RAISE_EXCEPTION (BMI_ERRCRIT);

    if (strcmp (relop, BMI_IX_eq) == 0)
    {   if (strcmp (modifier, BMI_IX_deriv) == 0)
	    fleader = &bmi_test_leader_deriv_eq;
	else if (strcmp (modifier, BMI_IX_proper) == 0)
	    fleader = &bmi_test_leader_proper_eq;
	else
	    fleader = &bmi_test_leader_eq;
	frank = &bmi_test_rank_eq;
	forder = &bmi_test_order_eq;
    } else if (strcmp (relop, BMI_IX_ne) == 0)
    {   if (strcmp (modifier, BMI_IX_deriv) == 0)
	    fleader = &bmi_test_leader_deriv_ne;
	else if (strcmp (modifier, BMI_IX_proper) == 0)
	    fleader = &bmi_test_leader_proper_ne;
	else
	    fleader = &bmi_test_leader_ne;
	frank = &bmi_test_rank_ne;
	forder = &bmi_test_order_ne;
    } else if (strcmp (relop, BMI_IX_gt) == 0)
    {	fleader = &bmi_test_leader_gt;
	frank = &bmi_test_rank_gt;
	forder = &bmi_test_order_gt;
    } else if (strcmp (relop, BMI_IX_ge) == 0)
    {   fleader = &bmi_test_leader_ge;
	frank = &bmi_test_rank_ge;
	forder = &bmi_test_order_ge;
    } else if (strcmp (relop, BMI_IX_lt) == 0)
    {	fleader = &bmi_test_leader_lt;
	frank = &bmi_test_rank_lt;
	forder = &bmi_test_order_lt;
    } else if (strcmp (relop, BMI_IX_le) == 0)
    {	fleader = &bmi_test_leader_le;
	frank = &bmi_test_rank_le;
	forder = &bmi_test_order_le;
    } else
	BA0_RAISE_EXCEPTION (BMI_ERRCRIT);

    if (leader)
    {	for (i = T->size - 1; i >= 0; i--)
	{   bav_variable u;
	    bool b;

	    b = integer ? 
		bap_is_numeric_polynom_mpz ((bap_polynom_mpz)T->tab [i]) :
		bap_is_numeric_polynom_mpq ((bap_polynom_mpq)T->tab [i]);
	    u = b ? BAV_NOT_A_VARIABLE : 
			integer ?
			bap_leader_polynom_mpz ((bap_polynom_mpz)T->tab [i]) :
			bap_leader_polynom_mpq ((bap_polynom_mpq)T->tab [i]);
	    if (! (*fleader) (v, b, u))
		ba0_delete_table (T, i);
	}
    } else if (order)
    {	for (i = T->size - 1; i >= 0; i--)
	{   bav_Iorder p;
	    bool b;

	    b = integer ?
		bap_is_independent_polynom_mpz ((bap_polynom_mpz)T->tab [i]) :
		bap_is_independent_polynom_mpq ((bap_polynom_mpq)T->tab [i]);
	    p = b ? 0 : 
		    integer ?
		    bap_total_order_polynom_mpz ((bap_polynom_mpz)T->tab [i]) :
		    bap_total_order_polynom_mpq ((bap_polynom_mpq)T->tab [i]);
	    if (! (*forder) (o, b, p))
		ba0_delete_table (T, i);
	}
    } else
    {	for (i = T->size - 1; i >= 0; i--)
	{   struct bav_rank rk;

	    rk = integer ?
		bap_rank_polynom_mpz ((bap_polynom_mpz)T->tab [i]) :
		bap_rank_polynom_mpq ((bap_polynom_mpq)T->tab [i]);
	    if (! (*frank) (&rg, &rk))
		ba0_delete_table (T, i);
	}
    }

    bav_set_settings_symbol (0, &bav_printf_numbered_symbol);
    if (T->size == 0)
	buffer = ba0_strdup ("[]");
    else if (rw)
    {   ba0_record_output ();
        ba0_set_output_counter ();
	if (integer)
            bmi_printf_rewrite_rules_mpz 
				((bap_tableof_polynom_mpz)T, '[', ']');
	else
	    bmi_printf_rewrite_rules_mpq
				((bap_tableof_polynom_mpq)T, '[', ']');
        buffer = ba0_persistent_malloc (ba0_output_counter () + 1);
        ba0_set_output_string (buffer);
	if (integer)
            bmi_printf_rewrite_rules_mpz 
				((bap_tableof_polynom_mpz)T, '[', ']');
	else
	    bmi_printf_rewrite_rules_mpq
				((bap_tableof_polynom_mpq)T, '[', ']');
        ba0_restore_output ();
    } else
	buffer = ba0_new_printf (integer ? "%t[%Az]" : "%t[%Aq]", T);

    {   ALGEB res;
	bmi_push_maple_gmp_allocators ();
        res = EvalMapleStatement (callback->kv, buffer);
	bmi_pull_maple_gmp_allocators ();
        return res;
    }
}

ALGEB bmi_equations_with_criterion (struct bmi_callback* callback)
{
    return bmi__equations_with_criterion (callback, false);
}

ALGEB bmi_rewrite_rules_with_criterion (struct bmi_callback* callback) 
{
    return bmi__equations_with_criterion (callback, true);
}

