#include "bmi_indices.h"
#include "bmi_callback.h"
#include "bmi_memory.h"
#include "bmi_options.h"
#include "bmi_exported.h"
#include "bmi_blad_eval.h"
#include "bmi_mesgerr.h"
#include "bmi_gmp.h"

static char bmi_mesgerr [BMI_BUFSIZE];

/*
 * Returns true if the BLAD libraries are successfully called.
 * In this case, the computation result, an ALGEB, is stored in *result.
 */

static bool bmi_process_command (ALGEB* result, struct bmi_options* options, 
				 struct bmi_callback* callback)
{   struct ba0_exception_code code;
    ba0_scanf_function* s;
    ba0_printf_function* f;
    char* name;
/*
 * Sequence of calls to the BLAD libraries
 * Increasing the default cell size of ba0_analex and the default token_lmax
 * The leading string for parsing orderings is now "ranking"
 */
    ba0_set_settings_no_oot (true);
    ba0_set_settings_no_oom (true);
    if (strcmp (options->cellsize, BMI_IX_small) == 0)
	ba0_set_settings_stack (0x10000, 0, 0x10000, 0, 0);
    else
	ba0_set_settings_stack (0, 0, 0x10000, 0, 0);
    bav_set_settings_ordering ("ranking");

    bai_restart (options->time_limit, options->memory_limit);
/*
 * If an error occurs in BLAD, an exception is raised, which is
 * caught here. The error message is enhanced with a context string
 * when the error is raised by a parser.
 */
    BA0_PUSH_EXCEPTION (code);
    if (ba0_exception_is_raised (code))
    {	char* context = ba0_get_context_analex ();
	if (context [0])
	    sprintf (bmi_mesgerr, "%s (approx. error location: %s)", 
						ba0_mesgerr, context);
	else
	    sprintf (bmi_mesgerr, "%s", ba0_mesgerr);
	bai_terminate (ba0_init_level);
	return false;
    }
    ba0_set_settings_no_oot (false);
    ba0_set_settings_no_oom (false);
    bmi_check_blad_gmp_allocators (__FILE__, __LINE__);
/*
 * Customize the input notation
 */
    switch (options->input_notation)
    {	case bmi_jet_notation:
	case bmi_tjet_notation:
	    s = &bav_scanf_jet_variable;
	    break;
	case bmi_diff_notation:
	    s = &bav_scanf_diff_variable;
	    break;
	case bmi_udif_notation:
	    s = &bav_scanf_Diff_variable;
	    break;
	default: /* to avoid a stupid warning */
	    s = 0;
    }
/*
 * Customize the output notation
 */
    switch (options->output_notation)
    {	case bmi_jet_notation:
	    f = &bav_printf_jet_variable;
	    break;
	case bmi_tjet_notation:
	    f = &bav_printf_jet_wesb_variable;
	    break;
	case bmi_diff_notation:
	    f = &bav_printf_diff_variable;
	    break;
	case bmi_udif_notation:
	    f = &bav_printf_Diff_variable;
	    break;
	default: /* to avoid a stupid warning */
	    f = 0;
    }
    bav_set_settings_variable (s, f, 0);
    bav_set_settings_ordering ("ranking");

    bmi_check_blad_gmp_allocators (__FILE__, __LINE__);
/*
 * The string ("NormalForm", "RosenfeldGroebner", ...) representing 
 * the exported function to be called.
 */
    name = bmi_string_op (0, callback);

    bmi_check_blad_gmp_allocators (__FILE__, __LINE__);
/*
 * The call
 */
    *result = bmi_call_exported (name, callback);
/*
 * Exit from BLAD
 */
    bai_terminate (ba0_init_level);
    return true;
}

/*
 * The entry point
 */

BMI_DLL ALGEB M_DECL bmi_blad_eval (MKernelVector kv, ALGEB args)
{   struct bmi_options options;
    struct bmi_callback* callback;
    long nargs;
    ALGEB result;
    bool b;
/*
 * args is a sequence and one should not apply op and nops over it.
 */
    nargs = (long)MapleNumArgs (kv, args);
    if (nargs < 1)
        MapleRaiseError (kv, BMI_ERRNARGS);
/*
 * Call this one first 
 */
    bai_reset_all_settings ();
/*
 * Initialize the memory management data structure 
 * Returns a callback for calling MAPLE.
 */
    callback = bmi_init_memory (kv, ((ALGEB*)args) [1]);
/*
 * Process the options.
 */
    bmi_init_options (&options);
    if (! bmi_set_options_maple (&options, kv, (ALGEB*)args+2, nargs-1))
    {   bmi_clear_options (&options);
	bmi_clear_memory ();
	MapleRaiseError (kv, BMI_ERROPTS);
    }
/*
 * Process the command.
 */
    result = (ALGEB)0;
    b = bmi_process_command (&result, &options, callback);
/*
 * Clear the data structure 
 */
    bmi_clear_options (&options);
    bmi_clear_memory ();
/*
 * if b is false then result is not set.
 */
    if (! b)
    {	
#if defined (BMI_MEMCHECK)
	if (result)
	{   fprintf (stderr, "bmi fatal error: NULL expected\n");
	    exit (1);
	}
#endif
	MapleRaiseError (kv, bmi_mesgerr);
    }

    MapleGcAllow (kv, result);

    bmi_check_maple_gmp_allocators (__FILE__, __LINE__);
    bmi_check_gmp_sp ();
    bmi_check_error_sp ();

    return result;
}

